function validate(form_selector, fields, successCallbackFunction, successAjaxCallbackFunction, config) {
    var form = $(form_selector);

    config = config || {};

    var self = {
        form_selector: form_selector,
        form: form,
        form_parent: form.parent(),
        submit_button: form.find("button[type='submit']"),
        submit_url: (config.submit_url) ?? form.attr('action'),
        is_submitted: false,
        fields: fields,
        message_placeholder: form.parent().siblings('.form-message-placeholder').first(),
        has_custom_fields: (config.custom_fields) ? config.custom_fields.has_custom_fields : false,
        custom_fields_wrapper: (config.custom_fields) ? $(config.custom_fields.wrapper_id) : null,
        is_custom_fields_validate_passed: false,
        activation_modal: null,
        config: config
    };

    $.each(self.fields, function (index, field) {
        field.is_valid = false;
        field.error = false;

        if (field.parent === undefined && $.isFunction(field.input.closest)) {
            field.parent = field.input.closest('.form-group');
        }
        if (field.help_block === undefined && $.isFunction(field.input.siblings)) {
            field.help_block = field.input.siblings('.help-block').first();
        }
    });

    form.on('submit', function (e) {
        e.preventDefault();

        self.is_submitted = true;

        validateAllFields();

        if (isFormValid()) {
            if (successCallbackFunction) {
                successCallbackFunction(self);
            } else {
                sendFormData();
            }
        } else {
            $.each(self.fields, function (index, field) {
                applyErrorStatus(field);
            });

            errorToast(_e('FIX_ERRORS'));
            scrollToElement(self.form_selector);
        }
    });

    /**
     * Sends Form Data
     */
    function sendFormData() {

        self.addLoading();

        $.ajax({
            url: self.submit_url,
            dataType: 'json',
            type: 'POST',
            data: self.form.serialize(),
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-CSRF-TOKEN', self.getCsrfToken());
            },
            success: function (response) {

                if (response.status) {
                    if (successAjaxCallbackFunction) {
                        self.removeLoading();
                        successAjaxCallbackFunction(null, response, self);
                        return false;
                    }

                    if (response.message) {
                        successToast(response.message);
                    }

                    if (response.type === 'redirect') {
                        redirectTo(response.url);
                        return false;
                    }

                    self.removeLoading();

                    if (response.type === 'html') {
                        self.form.parent().slideUp(function () {
                            self.message_placeholder.html(response.html).slideDown();
                        });

                        scrollToElement('body');
                    }

                    if (response.type === 'form' && response.form) {
                        form.append(response.form);
                    }
                } else {
                    if (self.config.submitErrorCallback) {
                        self.removeLoading();
                        self.config.submitErrorCallback(response, self);
                        return false;
                    }
                    self.removeLoading();
                    self.resetCaptcha();
                    errorToast(response.errors);
                }
            }
        });
    }

    /**
     * Loop through all fields and validates them
     */
    function validateAllFields() {
        $.each(self.fields, function (index, field) {
            validateField(field);
        });

        if (self.has_custom_fields) {
            validateCustomFields();
        }
    }

    /**
     * Validates the given field
     *
     * @param field
     */
    function validateField(field) {
        if (field.validate.rule === 'username') {
            validateUsername(field);
        }
        if (field.validate.rule === 'password') {
            validatePassword(field);
        }
        if (field.validate.rule === 'password_confirm') {
            validatePasswordConfirm(field);
        }
        if (field.validate.rule === 'required') {
            validateRequired(field);
        }
        if (field.validate.rule === 'email') {
            validateEmailField(field);
        }
        if (field.validate.rule === 'mobile') {
            validateMobileField(field);
        }
        if (field.validate.rule === 'selectbox') {
            validateSelectboxField(field);
        }
        if (field.validate.rule === 'checkbox') {
            validateCheckboxField(field);
        }
        if (field.validate.rule === 'submit_auth_password') {
            validateSubmitAuthPassword(field);
        }
        if (field.validate.rule === 'category') {
            validateCategoryField(field);
        }
        if (field.validate.rule === 'location') {
            validateLocationField(field);
        }
        if (field.validate.rule === 'radio-group') {
            validateRadioGroupField(field);
        }
    }

    function validateCustomFields() {
        self.is_custom_fields_validate_passed = true;

        validatePriceCustomFields();
        validateTextCustomFields();
        validateRadioCustomFields();
        validateSelectCustomFields();
    }

    /**
     * Validates Username Rule
     *
     * @param field
     */
    function validateUsername(field) {
        var value = field.input.val();

        applyUsernameValidation(field, value);

        field.input.on('input', function () {
            applyUsernameValidation(field, $(this).val());
        })
    }

    /**
     * Applies the status of validation to username field
     *
     * @param field
     * @param value
     */
    function applyUsernameValidation(field, value) {
        if (field.validate.should_visible &&  ! field.input.is(':visible')) {
            makeValid(field);
        } else {
            if (GLOBE__username_type === 'email' && !isEmail(value)) {
                makeInvalid(field, _e('INVALID_EMAIL'));
            } else if (GLOBE__username_type === 'mobile' && !isMobile(value)) {
                makeInvalid(field, _e('INVALID_MOBILE'));
            } else if (GLOBE__username_type === 'all' && !isMobile(value) && !isEmail(value)) {
                makeInvalid(field, _e('INVALID_MOBILE_EMAIL'));
            } else {
                makeValid(field);
            }
        }
    }

    /**
     * Validates Password Rule
     */
    function validatePassword(field) {
        var value = field.input.val();

        applyPasswordValidation(field, value);
        field.input.on('input', function () {
            applyPasswordValidation(field, $(this).val());
        });
    }

    /**
     * Applies the status of validation to password field
     *
     * @param field
     * @param value
     */
    function applyPasswordValidation(field, value) {
        if (value.length < field.validate.min_length) {
            makeInvalid(field, _e('PASSWORD_MIN_CHARACTERS_VALIDATION', field.validate.min_length));
        } else {
            makeValid(field);
        }
    }

    /**
     * Validates Password Confirm Field
     */
    function validatePasswordConfirm(field) {
        var value = field.input.val();
        var parent_field = findField(field.validate.relates_to);

        applyPasswordConfirmValidation(field, value, parent_field);

        field.input.on('input', function () {
            applyPasswordConfirmValidation(field, $(this).val(), parent_field);
        });
    }

    /**
     * Applies the status of validation to password confirm field
     *
     * @param field
     * @param value
     * @param parent_field
     */
    function applyPasswordConfirmValidation(field, value, parent_field) {
        if (value != parent_field.input.val()) {
            makeInvalid(field, _e('PASSWORD_NOT_MATCH'));
        } else {
            makeValid(field);
        }
    }

    /**
     * Validates Required Rule
     */
    function validateRequired(field) {
        var value = field.input.val();

        applyRequiredValidation(field, value);

        field.input.on('input', function () {
            applyRequiredValidation(field, $(this).val());
        });
    }

    /**
     * Applies the status of validation to password field
     *
     * @param field
     * @param value
     */
    function applyRequiredValidation(field, value) {
        var min_length = (field.validate.min_length) ? field.validate.min_length : 1;
        var error = (min_length === 1) ? _e('REQUIRED_FIELD') : _e('MIN_CHARACTERS_VALIDATION', min_length);

        if (value.length < min_length) {
            makeInvalid(field, error, min_length);
        } else {
            makeValid(field);
        }
    }


    /**
     * Validates Selectbox Rule
     */
    function validateSelectboxField(field) {
        var value = field.input.val();

        applySelectboxFieldValidation(field, value);
        field.input.on('change', function () {
            applySelectboxFieldValidation(field, $(this).val());
        });
    }

    /**
     * Applies the status of validation to selectbox field
     *
     * @param field
     * @param value
     */
    function applySelectboxFieldValidation(field, value) {
        if (!value || value === '' || value === 'any') {
            makeInvalid(field, _e('REQUIRED_FIELD'));
        } else {
            makeValid(field);
        }
    }

    /**
     * Validates Checkbox Rule
     */
    function validateCheckboxField(field) {
        var value = field.input.val();

        applyCheckboxFieldValidation(field, value);
        field.input.on('change', function () {
            applyCheckboxFieldValidation(field, $(this).val());
        });
    }

    /**
     * Applies the status of validation to Checkbox field
     *
     * @param field
     * @param value
     */
    function applyCheckboxFieldValidation(field, value) {
        if (!field.input.is(":checked")) {
            makeInvalid(field, _e('ACCEPT_TERMS_REQUIRED'));
        } else {
            makeValid(field);
        }
    }

    /**
     * Validates Checkbox Rule
     */
    function validateRadioGroupField(field) {
        var count = $(`${field.selector}:checked`).length;

        applyRadioGroupFieldValidation(field, count);
        field.input.on('change', function () {
            applyRadioGroupFieldValidation(field, $(`${field.selector}:checked`).length);
        });
    }

    /**
     * Applies the status of validation to Checkbox field
     *
     * @param field
     * @param count
     */
    function applyRadioGroupFieldValidation(field, count) {
        if (! count) {
            console.log('yessss');
            makeInvalid(field, _e('RADIO_GROUP_REQUIRED'));
        } else {
            makeValid(field);
        }
    }


    /**
     * Validates Email Rule
     */
    function validateEmailField(field) {
        var value = field.input.val();

        applyEmailFieldValidation(field, value);
        field.input.on('input', function () {
            applyEmailFieldValidation(field, $(this).val());
        });
    }

    /**
     * Applies the email validation field
     *
     * @param field
     * @param value
     */
    function applyEmailFieldValidation(field, value) {
        if (!value || value === '' || value === ' ' || !isEmail(value)) {
            makeInvalid(field, _e('INVALID_EMAIL'));
        } else {
            makeValid(field);
        }
    }

    /**
     * Validates Email Rule
     */
    function validateMobileField(field) {
        var value = field.input.val();

        applyMobileFieldValidation(field, value);
        field.input.on('input', function () {
            applyMobileFieldValidation(field, $(this).val());
        });
    }

    /**
     * Applies the email validation field
     *
     * @param field
     * @param value
     */
    function applyMobileFieldValidation(field, value) {
        if (!value || value === '' || value === ' ' || !isMobile(value)) {
            makeInvalid(field, _e('INVALID_MOBILE'));
        } else {
            makeValid(field);
        }
    }


    /**
     * Validates Password for auth section in submit page
     */
    function validateSubmitAuthPassword(field) {
        var value = field.input.val();

        applySubmitAuthPasswordValidation(field, value);
        field.input.on('input', function () {
            applySubmitAuthPasswordValidation(field, $(this).val());
        });
    }

    /**
     * Applies the status of validation to Password field for auth section in submit page
     *
     * @param field
     * @param value
     */
    function applySubmitAuthPasswordValidation(field, value) {
        if (!field.input.is(':visible') && !field.validate.needs_auth_password) {
            makeValid(field);
        } else {
            if (value === '' || value === ' ' || !value) {
                makeInvalid(field, _e('PASSWORD_REQUIRED'));
            } else {
                makeValid(field);
            }
        }
    }

    /**
     * Validates dynamic category select field
     */
    function validateCategoryField(field) {
        applyCategoryFieldValidation(field, field.input.subcategory.val());
        field.input.subcategory.on('input', function () {
            applyCategoryFieldValidation(field, $(this).val());
        });
    }

    /**
     * Applies validation for dynamic category select field
     *
     * @param field
     * @param subcategory_value
     */
    function applyCategoryFieldValidation(field, subcategory_value) {
        if (!subcategory_value || subcategory_value === '' || subcategory_value === 0 || subcategory_value === 'any') {
            makeInvalid(field, _e('CATEGORY_REQUIRED'));
        } else {
            makeValid(field);
        }
    }


    /**
     * Validates dynamic category select field
     */
    function validateLocationField(field) {
        applyLocationFieldValidation(field, field.input.city.val());

        field.input.city.on('input', function () {
            applyLocationFieldValidation(field, $(this).val());
        });
    }

    /**
     * Applies validation for dynamic category select field
     *
     * @param field
     * @param city_value
     */
    function applyLocationFieldValidation(field, city_value) {
        if (!city_value || city_value === '' || city_value === 0 || city_value === 'any') {
            makeInvalid(field, _e('LOCATION_REQUIRED'));
        } else {
            makeValid(field);
        }
    }

    /**
     * Validates Price Custom Field
     */
    function validatePriceCustomFields() {
        self.custom_fields_wrapper.find('.custom-field__price').each(function (index, input) {

            var $this = $(input);

            if (!isCustomFieldRequired($this)) {
                return false;
            }

            var name = $this.attr('data-name');
            var price_type_input = $this.find('#' + name + '-type');
            var price_input = $this.find('#' + name);
            var price_type = price_type_input.val();
            var price = price_input.val();

            if (price_type !== 'price' && price_type !== 'free' && price_type !== 'deal' && price_type !== 'exchange') {
                makeCustomFieldInvalid(price_type_input, _e('INVALID_VALUE'));
            } else {
                makeCustomFieldValid(price_type_input);
            }

            if (price_type === 'price' && (!price || price === '' || price === ' ' || price === 0)) {
                makeCustomFieldInvalid(price_input);
            } else {
                makeCustomFieldValid(price_input);
            }
        });
    }

    /**
     * Validates Text Custom Field
     */
    function validateTextCustomFields() {

        self.custom_fields_wrapper.find('.custom-field__text').each(function (index, input) {
            var $this = $(input);

            if (!isCustomFieldRequired($this)) {
                return false;
            }

            var name = $this.attr('data-name');
            var field = $this.find("input[name=" + name + "]");
            if (field.val() === '' || field.val() === ' ') {
                makeCustomFieldInvalid(field);
            } else {
                makeCustomFieldValid(field);
            }
        });
    }

    /*
     * Validates Radio Custom Field
     */
    function validateRadioCustomFields() {

        self.custom_fields_wrapper.find('.custom-field__radio').each(function (index, input) {
            var $this = $(input);

            if (!isCustomFieldRequired($this)) {
                return false;
            }

            var name = $this.attr('data-name');
            if (!$this.find("input[name=" + name + "]:checked").val()) {
                makeRadioFieldInvalid($this);
            } else {
                makeRadioFieldValid($this);
            }
        });
    }

    function validateSelectCustomFields() {
        self.custom_fields_wrapper.find('.custom-field__select').each(function (index, input) {
            var $this = $(input);

            if (! isCustomFieldRequired($this)) {
                return false;
            }

            var name = $this.attr('data-name');
            var field = $this.find("select[name=" + name + "]");

            if (! field.val() || field.val() === '' || field.val() === 'any') {
                makeCustomFieldInvalid(field, _e('SELECT_VALUE'));
            } else {
                makeCustomFieldValid(field);
            }
        });
    }

    /*
    * finds and returns the field with the given id from self.fields
    */
    function findField(field_id) {

        var field_index = null;

        $.each(self.fields, function (index, field) {
            if (field.id == field_id) {
                field_index = index;
                return false;
            }
        });

        return self.fields[field_index];
    }

    /*
    * flags the given field as invalid
    */
    function makeInvalid(field, error) {
        field.is_valid = false;
        field.error = error;
        if (self.is_submitted) {
            applyErrorStatus(field);
        }
    }

    /*
    * flags the given field as valid
    */
    function makeValid(field) {
        field.is_valid = true;
        field.error = null;

        if (self.is_submitted) {
            applyErrorStatus(field);
        }
    }

    /*
    * flags the given custom field as valid
    */
    function makeCustomFieldValid(input) {
        input.removeClass('error');
        input.closest('.form-group').removeClass('has-error');
        input.siblings('.help-block').first().removeClass('form-error').html('');
    }

    /*
    * flags the given custom field as invalid
    */
    function makeCustomFieldInvalid(input, error) {
        error = error || _e('REQUIRED_FIELD');

        self.is_custom_fields_validate_passed = false;

        input.addClass('error');
        input.closest('.form-group').addClass('has-error');
        input.siblings('.help-block').first().addClass('form-error').html(error);
    }

    /*
    * flags the given Radio field as valid
    */
    function makeRadioFieldValid(input) {
        input.find('.form-group').removeClass('has-error');
        input.find('.help-block').removeClass('form-error').html('');
    }

    /*
    * flags the given Radio field as invalid
    */
    function makeRadioFieldInvalid(input) {
        self.is_custom_fields_validate_passed = false;

        input.find('.form-group').addClass('has-error');
        input.find('.help-block').addClass('form-error').html(_e('SELECT_VALUE'));
    }

    /*
    * Shows or hides error status for the given field based on the current condition of field
    */
    function applyErrorStatus(field) {
        if (!field.is_valid) {
            if ($.isFunction(field.input.addClass)) {
                field.input.addClass('error');
            }
            if ($.isFunction(field.parent.addClass)) {
                field.parent.addClass('has-error');
            }
            if ($.isFunction(field.help_block.addClass)) {
                field.help_block.addClass('form-error').html(field.error);
            }
        } else {
            if ($.isFunction(field.input.removeClass)) {
                field.input.removeClass('error');
            }
            if ($.isFunction(field.parent.removeClass)) {
                field.parent.removeClass('has-error');
            }
            if ($.isFunction(field.help_block.removeClass)) {
                field.help_block.removeClass('form-error').html('');
            }
        }
    }

    /*
    * Checks whether the form is valid or not
    */
    function isFormValid() {
        var is_valid = true;

        if (self.has_custom_fields && ! customFieldsValidatePassed()) {
            is_valid = false;
        }

        $.each(self.fields, function (index, field) {
            if (!field.is_valid) {
                is_valid = false;
                return false;
            }
        });

        return is_valid;
    }

    function customFieldsValidatePassed() {
        return self.is_custom_fields_validate_passed;
    }

    function isCustomFieldRequired(input) {
        return input.attr('data-is-required') == 1;
    }

    /**
     * Returns the value of csrf token field
     */
    self.getCsrfToken = function () {
        return form.find("input[name='token']").val();
    };

    /*
    * adds the loading to page and elements
    */
    self.addLoading = function () {
        addButtonLoading(self.submit_button);
    };

    /*
    * removes loading from page and elements
    */
    self.removeLoading = function () {
        removeButtonLoading(self.submit_button);
    };

    /**
     * Resets the captcha based on the type of captcha
     *
     */
    self.resetCaptcha = function () {
        if (isGoogleRecaptcha()) {
            grecaptcha.reset();
        }
        if (isImageCaptcha()) {
            self.form.find('.captcha-update-handler i').trigger('click');
        }
    }

    $(document).on("authActivationSuccess", function (e) {
        sendFormData();
    });

    $(document).on("modalConfirmSuccess", function (e) {
        e.modal.close();
        sendFormData();
    });
}