/**
 * Creates a simple Map
 *
 * @param map_canvas_id
 * @param latitude
 * @param longitude
 * @param zoom
 * @param draggable_marker
 * @param title
 */
function simpleMap(map_canvas_id, latitude, longitude, zoom, draggable_marker, title) {
    title = title || '';
    draggable_marker = draggable_marker || false;

    if (isGoogleMaps()) {
        google.maps.event.addDomListener(window, 'load', function() {
            simpleGoogleMap(map_canvas_id, latitude, longitude, zoom, title);
        });
    }

    if (isCedarMaps() || isOSMMaps() || isNeshanMaps()) {
        return simpleLeafletMap(map_canvas_id, latitude, longitude, zoom, draggable_marker);
    }
}

/**
 * Creates a simple Map via leafletJs
 *
 * @param map_canvas_id
 * @param latitude
 * @param longitude
 * @param zoom
 * @param draggable_marker
 */
function simpleLeafletMap(map_canvas_id, latitude, longitude, zoom, draggable_marker) {

    if (isNeshanMaps()) {
        var custom_map = new L.Map(document.getElementById(map_canvas_id), {
            maptype: 'dreamy',
            key: GLOBE__neshan_token,
            center: [latitude, longitude],
            zoom: zoom
        });
    } else {
        var custom_map = L.map(map_canvas_id, {attributionControl: false}).setView([latitude, longitude], zoom);
    }

    draggable_marker = draggable_marker || false;

    if (GLOBE__maps_source === 'CEDAR') {
        L.tileLayer('https://api.cedarmaps.com/v1/tiles/{id}/{z}/{x}/{y}.png?access_token={accessToken}', {
            id: 'cedarmaps.streets',
            accessToken: GLOBE__cedar_token
            //maxZoom: zoom + 5,
        }).addTo(custom_map);
    }

    if (GLOBE__maps_source === 'OSM') {
        L.tileLayer('http://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
            attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a>',
            subdomains: ['a','b','c']
        }).addTo(custom_map);
    }

    var marker = L.marker([latitude, longitude], {icon: mapMarker(), draggable: draggable_marker});
    // marker.bindPopup( '' );
    marker.addTo(custom_map);

    return {map: custom_map, marker: marker};
}

function simpleGoogleMap(map_canvas_id, latitude, longitude, zoom, title) {
    var coordinate = new google.maps.LatLng(latitude, longitude);
    var mapOptions = {
        zoom: zoom,
        center: coordinate
    };

    var custom_map = new google.maps.Map(document.getElementById(map_canvas_id), mapOptions);
    new google.maps.Marker({
        position: coordinate,
        map: custom_map,
        title: title
    });
}

function initializeSimpleGoogleMap() {

}

/**
 * adds marker to map
 */
function mapMarker() {
    return createLeafletMapMarker();
}

/**
 * adds marker to leafletJs map
 */
function createLeafletMapMarker() {
    return L.icon({
        iconUrl:      GLOBE__site_url + 'images/leaflet-marker.png',
        shadowSize:   [26, 42], // size of the shadow
        iconAnchor:   [13, 42], // point of the icon which will correspond to marker's location
        popupAnchor:  [0, -42] // point from which the popup should open relative to the iconAnchor
    });
}

/**
 * Initialize GoogleMaps Location Picker
 *
 * @param map_canvas_id
 * @param latitude
 * @param longitude
 * @param zoom
 */
function initializeGoogleLocationPicker(map_canvas_id, latitude, longitude, zoom) {
    showLocationPicker(map_canvas_id, {
        'default_lat' : latitude,
        'default_lng' : longitude,
        'zoom': zoom
    });

    $(document).on('locationUpdated', function(e) {
        var latitude = e.lat;
        var longitude = e.long;

        if (! latitude || ! longitude || latitude == 'null' || longitude == 'null') {
            return false;
        }

        if (! latitude || ! longitude || latitude == 'null' || longitude == 'null') {
            return false;
        }

        var map_selector_id = $(this).attr('data-map-selector-id');
        updateGoogleLocationPicker(map_selector_id, latitude, longitude);
    });
}

/**
 * Updates GoogleMaps Location Picker
 *
 * @param map_selector_id
 * @param latitude
 * @param longitude
 */
function updateGoogleLocationPicker(map_selector_id, latitude, longitude) {
    $("#" + map_selector_id).locationpicker("location", {latitude: latitude, longitude: longitude});
}

/**
 * Initialize leafletMap Location Picker
 *
 * @param map_canvas_id
 * @param latitude
 * @param longitude
 * @param zoom
 */
function initializeLeafletLocationPicker(map_canvas_id, latitude, longitude, zoom) {
    var location_picker = simpleLeafletMap(map_canvas_id, latitude, longitude, zoom, true);

    bindLeafletMarkerDragEvent(location_picker.marker);

    $(document).on('locationUpdated', function(e) {
        console.log('yess');
        var latitude = e.lat;
        var longitude = e.long;

        if (! latitude || ! longitude || latitude == 'null' || longitude == 'null') {
            return false;
        }

        location_picker = updateLeafletLocationPicker(location_picker.map, latitude, longitude, location_picker.marker, zoom);
        bindLeafletMarkerDragEvent(location_picker.marker);
    });
}

/**
 * Updates leafletMap Location Picker
 *
 * @param leaflet_map
 * @param latitude
 * @param longitude
 * @param leaflet_marker
 * @param zoom
 * @returns {{map: *, marker: *}}
 */
function updateLeafletLocationPicker(leaflet_map, latitude, longitude, leaflet_marker, zoom) {

    var newLatLng = new L.LatLng(latitude, longitude);

    leaflet_marker.setLatLng(newLatLng);

    leaflet_map.setView([latitude, longitude], zoom);

    updateLeafletHiddenInputs(latitude, longitude);

    return {map: leaflet_map, marker: leaflet_marker};
}

/**
 * Create and initialise Advanced Leaflet Location picker (with search)
 *
 * @param map_canvas_id
 * @param latitude
 * @param longitude
 * @param zoom
 */
function initialiseAdvancedLeafletLocationPicker(map_canvas_id, latitude, longitude, zoom) {
    var location_picker = simpleLeafletMap(map_canvas_id, latitude, longitude, zoom, true);

    bindLeafletMarkerDragEvent(location_picker.marker);
    updateLeafletHiddenInputs(latitude, longitude);

    var search_input = $("#locationName");
    var typingTimer;

    var doneTypingInterval = 300;
    search_input.on('input keyup', function () {
        clearTimeout(typingTimer);
        typingTimer = setTimeout(function() {
            leafletSearchCallback(search_input, location_picker)
        }, doneTypingInterval);
    });
    search_input.on('keydown', function () {
        clearTimeout(typingTimer);
    });

    $(document).on('locationUpdated', function(e) {
        var latitude = e.lat;
        var longitude = e.long;

        if (! latitude || ! longitude || latitude == 'null' || longitude == 'null') {
            return false;
        }

        location_picker = updateLeafletLocationPicker(location_picker.map, latitude, longitude, location_picker.marker, zoom);
        bindLeafletMarkerDragEvent(location_picker.marker);
    });
}

function leafletSearchCallback(search_input, leaflet_object) {
    var search_param = search_input.val();
    var search_result_placeholder = $("#map__search_result");
    var loading = $("#map_search__loading");

    search_result_placeholder.html('').slideUp();
    loading.hide();

    if (! search_param) {
        return false;
    }

    loading.show();
    $.ajax({
        url : GLOBE__site_url + 'ajax/map-search',
        dataType : 'json',
        type : 'post',
        data : {
            q: search_input.val()
        },
        success : function(response){
            if (response.status) {
                var html = "";
                response.result.forEach(function(item) {
                    html += "<div class='map_search__item' data-latitude='"+ item.loc[0] +"' data-longitude='"+ item.loc[1] +"'>"+ item.title +"</div>"
                });

                search_result_placeholder.html(html).slideDown();
                loading.hide();
            }
        }
    });

    $('body').on('click', ".map_search__item", function() {
        var $this = $(this);

        leaflet_object = updateLeafletLocationPicker(leaflet_object.map, $this.attr('data-latitude'), $this.attr('data-longitude'), leaflet_object.marker, 15);

        search_result_placeholder.html('').slideUp();

        $("#locationName").val(stripTags($this.html()));
    });
}

/**
 * Bind drag event to leafletMaps marker
 * @param marker
 */
function bindLeafletMarkerDragEvent(marker) {
    marker.on('dragend', function(e) {
        updateLeafletHiddenInputs(e.target._latlng.lat, e.target._latlng.lng);
    });
}

/**
 * Updates Map
 *
 * @param latitude
 * @param longitude
 */
function updateLeafletHiddenInputs(latitude, longitude) {
    $("#frm_latitude").val(latitude);
    $("#frm_longitude").val(longitude);
}